<?php

namespace App\Http\Controllers;

use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\Sale;
use App\Models\Category;
use App\Models\BusinessConfig;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function dailySales(Request $request)
    {
        $date = $request->input('date', Carbon::today()->format('Y-m-d'));

        $sales = Sale::with('details.product', 'user')
            ->whereDate('created_at', $date)
            ->where('is_internal', false)
            ->get();

        $internalSales = Sale::with('details.product', 'user')
            ->whereDate('created_at', $date)
            ->where('is_internal', true)
            ->get();

        $totalSales = $sales->sum('total');
        $totalInternal = $internalSales->sum('total');

        $paymentMethods = $sales->groupBy('payment_method')->map(function ($group) {
            return $group->sum('total');
        });

        return view('admin.reports.daily_sales', compact(
            'sales',
            'internalSales',
            'totalSales',
            'totalInternal',
            'paymentMethods',
            'date'
        ));
    }

    public function inventory()
    {
        $lowStockProducts = Product::whereColumn('stock', '<=', 'min_stock')->get();

        $stockValue = Product::select(DB::raw('SUM(stock * sale_price) as total_value'))->value('total_value');

        $movements = InventoryMovement::with('product', 'user')->latest()->take(50)->get();

        return view('admin.reports.inventory', compact('lowStockProducts', 'stockValue', 'movements'));
    }

    public function productSales(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::today()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::today()->format('Y-m-d'));

        $productSales = DB::table('sale_details')
            ->join('sales', 'sale_details.sale_id', '=', 'sales.id')
            ->join('products', 'sale_details.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->select(
                'products.name as product_name',
                'categories.name as category_name',
                DB::raw('SUM(sale_details.quantity) as total_quantity'),
                DB::raw('AVG(sale_details.price) as avg_price'),
                DB::raw('SUM(sale_details.subtotal) as total_revenue')
            )
            ->whereBetween('sales.created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59'])
            ->groupBy('products.id', 'products.name', 'categories.name')
            ->orderBy('total_revenue', 'desc')
            ->get();

        $grandTotal = $productSales->sum('total_revenue');

        return view('admin.reports.product_sales', compact('productSales', 'grandTotal', 'startDate', 'endDate'));
    }

    public function inventoryPrint()
    {
        $categories = Category::with([
            'products' => function ($query) {
                $query->where('is_active', true)->orderBy('name');
            }
        ])->orderBy('name')->get();

        $config = BusinessConfig::first();

        return view('admin.reports.inventory-print', compact('categories', 'config'));
    }
}
