<?php

namespace App\Http\Controllers;

use App\Models\Impresora;
use Illuminate\Http\Request;

class PrinterController extends Controller
{
    public function index()
    {
        $printers = Impresora::all();
        return view('admin.printers.index', compact('printers'));
    }

    public function create()
    {
        return view('admin.printers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:windows,network,usb',
            'ip' => 'required_if:tipo,network|nullable|string',
            'puerto' => 'required_if:tipo,network|integer',
            'es_predeterminada' => 'boolean',
            'use_browser' => 'boolean',
        ]);

        if ($request->has('es_predeterminada') && $request->es_predeterminada) {
            Impresora::where('es_predeterminada', true)->update(['es_predeterminada' => false]);
        }

        Impresora::create($validated);

        return redirect()->route('printers.index')->with('success', 'Impresora registrada exitosamente.');
    }

    public function edit(Impresora $printer)
    {
        return view('admin.printers.edit', compact('printer'));
    }

    public function update(Request $request, Impresora $printer)
    {
        $validated = $request->validate([
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:windows,network,usb',
            'ip' => 'required_if:tipo,network|nullable|string',
            'puerto' => 'required_if:tipo,network|integer',
            'es_predeterminada' => 'boolean',
            'activa' => 'boolean',
            'use_browser' => 'boolean',
        ]);

        if ($request->has('es_predeterminada') && $request->es_predeterminada) {
            Impresora::where('id', '!=', $printer->id)->update(['es_predeterminada' => false]);
        }

        $printer->update($validated);

        return redirect()->route('printers.index')->with('success', 'Impresora actualizada.');
    }

    public function destroy(Impresora $printer)
    {
        $printer->delete();
        return redirect()->route('printers.index')->with('success', 'Impresora eliminada.');
    }

    public function setDefault(Impresora $printer)
    {
        Impresora::where('es_predeterminada', true)->update(['es_predeterminada' => false]);
        $printer->update(['es_predeterminada' => true, 'activa' => true]);
        return back()->with('success', 'Impresora predeterminada establecida.');
    }

    public function discover()
    {
        $discovered = [];

        // 1. Detect Windows Local Printers (System & USB)
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $output = shell_exec('wmic printer get Name,PortName,DriverName');
            $lines = explode("\n", trim($output));
            array_shift($lines); // Remove header
            foreach ($lines as $line) {
                if (trim($line)) {
                    $parts = preg_split('/\s{2,}/', trim($line));
                    if (count($parts) >= 1) {
                        $discovered[] = [
                            'nombre' => $parts[1] ?? $parts[0],
                            'tipo' => (strpos(strtolower($parts[2] ?? ''), 'usb') !== false) ? 'usb' : 'windows',
                            'ip' => null,
                            'puerto' => null,
                            'detalles' => "Driver: " . ($parts[0] ?? 'N/A') . " | Puerto: " . ($parts[2] ?? 'N/A'),
                        ];
                    }
                }
            }
        }

        // 2. Scan Network for common Thermal Printer ports (9100)
        // For efficiency, we scan the server's subnet (simplified scan of last octet)
        $localIP = gethostbyname(gethostname());
        if ($localIP && $localIP !== '127.0.0.1') {
            $baseIP = preg_replace('/\.\d+$/', '.', $localIP);
            // Scan a small range or common POS static IPs (e.g., .100 to .150)
            // Note: PHP scanning is synchronous and slow. We'll limit it for the demo.
            foreach ([100, 101, 200, 201] as $suffix) {
                $ip = $baseIP . $suffix;
                $connection = @fsockopen($ip, 9100, $errno, $errstr, 0.1);
                if (is_resource($connection)) {
                    $discovered[] = [
                        'nombre' => "Impresora de Red ($ip)",
                        'tipo' => 'network',
                        'ip' => $ip,
                        'puerto' => 9100,
                        'detalles' => "Encontrada en puerto 9100",
                    ];
                    fclose($connection);
                }
            }
        }

        return view('admin.printers.discover', compact('discovered'));
    }
}
