<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Models\Caja;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\InventoryMovement;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\ActivityLog;

class POSController extends Controller
{
    public function index()
    {
        $categories = Category::with([
            'products' => function ($query) {
                $query->where('is_active', true);
            }
        ])->get();

        // Initial products (active only)
        $allProducts = Product::where('is_active', true)->limit(20)->get();

        return view('pos.index', compact('categories', 'allProducts'));
    }

    public function search(Request $request)
    {
        $query = $request->get('q');

        $products = Product::where('is_active', true)
            ->where(function ($q) use ($query) {
                $q->where('name', 'LIKE', "%{$query}%")
                    ->orWhere('barcode', 'LIKE', "%{$query}%");
            })
            ->with('category')
            ->limit(20)
            ->get();

        return response()->json($products);
    }

    public function store(Request $request)
    {
        $request->validate([
            'cart' => 'required|array|min:1',
            'cart.*.id' => 'required|exists:products,id',
            'cart.*.quantity' => 'required|integer|min:1',
            'payment_method' => 'required|in:cash,card,transfer,other',
            'amount_paid' => 'nullable|numeric|min:0',
        ]);

        // Check for open cash register
        $caja = Caja::where('is_active', true)->first();
        if (!$caja || !$caja->currentApertura) {
            return response()->json(['message' => 'No hay una caja abierta. Por favor abra una caja primero.'], 403);
        }
        $apertura = $caja->currentApertura;

        try {
            DB::beginTransaction();

            // Calculate Total
            $total = 0;
            $items = [];

            foreach ($request->cart as $item) {
                $product = Product::lockForUpdate()->find($item['id']);

                $subtotal = $product->sale_price * $item['quantity'];
                $total += $subtotal;

                $items[] = [
                    'product' => $product,
                    'quantity' => $item['quantity'],
                    'price' => $product->sale_price,
                    'subtotal' => $subtotal,
                ];
            }

            // Calculate Change
            $amountPaid = $request->amount_paid ?? $total;
            if ($request->payment_method === 'cash' && $amountPaid < $total) {
                throw new \Exception("Monto pagado insuficiente.");
            }
            $changeGiven = max(0, $amountPaid - $total);

            // Create Sale
            $sale = Sale::create([
                'apertura_caja_id' => $apertura->id,
                'user_id' => Auth::id(),
                'total' => $total,
                'payment_method' => $request->payment_method,
                'amount_paid' => $amountPaid,
                'change_given' => $changeGiven,
                'is_internal' => false,
            ]);

            // Process Items
            foreach ($items as $item) {
                // Deduct Stock
                $item['product']->decrement('stock', $item['quantity']);

                // Create Sale Detail
                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $item['product']->id,
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'subtotal' => $item['subtotal'],
                ]);

                // Record Inventory Movement
                InventoryMovement::create([
                    'product_id' => $item['product']->id,
                    'user_id' => Auth::id(),
                    'type' => 'venta',
                    'quantity' => -$item['quantity'], // Negative for sale
                    'current_stock_snapshot' => $item['product']->stock,
                    'reason' => "Venta #{$sale->id}",
                ]);
            }

            // Log Activity
            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'sale',
                'description' => "Venta #{$sale->id} registrada. Total: {$total}",
                'ip_address' => $request->ip(),
            ]);

            DB::commit();

            // Trigger silent printing
            try {
                $printService = app(\App\Services\PrintService::class);
                $printService->printSale($sale);
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error("Quick sale printing failed: " . $e->getMessage());
            }

            $responseData = [
                'success' => true,
                'message' => 'Venta registrada correctamente.',
                'sale_id' => $sale->id,
                'change' => $changeGiven,
            ];

            $defaultPrinter = \App\Models\Impresora::where('es_predeterminada', true)->first();
            if ($defaultPrinter && $defaultPrinter->use_browser) {
                $responseData['print_order'] = $sale->id;
                $responseData['print_type'] = 'sale';
            }

            return response()->json($responseData);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => $e->getMessage()], 500);
        }
    }

    public function printSaleTicket(Sale $sale)
    {
        $sale->load(['details.product', 'user']);
        $config = \App\Models\BusinessConfig::first() ?: new \App\Models\BusinessConfig([
            'business_name' => 'Sukha Cigar Bar & Lounge',
            'rnc' => '123-45678-9',
            'phone' => '(809) 555-0199',
            'address' => 'Av. Principal #123'
        ]);
        $tipo = 'factura';
        return view('pos.print-ticket-sale', compact('sale', 'tipo', 'config'));
    }
}
