<?php

namespace App\Http\Controllers;

use App\Models\Mesa;
use App\Models\Zona;
use Illuminate\Http\Request;

class MesaManagementController extends Controller
{
    /**
     * Display a listing of zones and tables.
     */
    public function index()
    {
        $zonas = Zona::with('mesas')->orderBy('orden')->get();
        // Tables without zone
        $mesasSinZona = Mesa::whereNull('zona_id')->get();

        return view('admin.mesas.index', compact('zonas', 'mesasSinZona'));
    }

    /**
     * Store a new zone.
     */
    public function storeZona(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'orden' => 'nullable|integer',
        ]);

        Zona::create($request->all());

        return back()->with('success', 'Zona creada correctamente.');
    }

    /**
     * Store a new table.
     */
    public function storeMesa(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'zona_id' => 'nullable|exists:zonas,id',
            'capacity' => 'nullable|integer|min:1',
            'is_internal' => 'nullable|boolean',
        ]);

        Mesa::create([
            'name' => $request->name,
            'zona_id' => $request->zona_id,
            'capacity' => $request->capacity ?? 4,
            'is_internal' => $request->has('is_internal'),
            'status' => 'libre',
            'is_active' => true
        ]);

        return back()->with('success', 'Mesa creada y asignada a la zona.');
    }

    /**
     * Update a table (name, capacity, or zone).
     */
    public function updateMesa(Request $request, Mesa $mesa)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'zona_id' => 'nullable|exists:zonas,id',
            'capacity' => 'nullable|integer|min:1',
            'is_internal' => 'nullable|boolean',
        ]);

        $data = $request->all();
        $data['is_internal'] = $request->has('is_internal');

        $mesa->update($data);

        return back()->with('success', 'Mesa actualizada correctamente.');
    }

    /**
     * Remove a zone.
     */
    public function destroyZona(Zona $zona)
    {
        // Tables in this zone will have zona_id set to null (handled by migration onDelete set null)
        $zona->delete();
        return back()->with('success', 'Zona eliminada correctamente.');
    }

    /**
     * Remove a table.
     */
    public function destroyMesa(Mesa $mesa)
    {
        if ($mesa->activeOrder) {
            return back()->with('error', 'No se puede eliminar una mesa con una orden activa.');
        }

        $mesa->delete();
        return back()->with('success', 'Mesa eliminada correctamente.');
    }
}
