<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\InventoryMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class InventoryController extends Controller
{
    public function index()
    {
        $movements = InventoryMovement::with(['product', 'user'])->latest()->paginate(20);
        return view('inventory.index', compact('movements'));
    }

    public function adjust(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'type' => 'required|in:entrada,salida,ajuste',
            'quantity' => 'required|integer|min:1',
            'reason' => 'nullable|string',
        ]);

        $product = Product::findOrFail($request->product_id);
        $quantity = $request->quantity;

        // Calculate new stock based on type
        if ($request->type === 'salida') {
            if ($product->stock < $quantity) {
                return back()->withErrors(['msg' => 'Stock insuficiente para esta salida.']);
            }
            $newStock = $product->stock - $quantity;
        } elseif ($request->type === 'entrada') {
            $newStock = $product->stock + $quantity;
        } else {
            // Ajuste (Set absolute value or difference? usually adjustment adds/subtracts to match physical count, 
            // but here let's assume 'ajuste' is a correction that mimics entry/exit but with specific reason. 
            // OR let's treat 'ajuste' as a direct stock set? 
            // For simplicity, let's treat 'ajuste' as a manual correction acting like entry/exit depending on context, 
            // but usually users want to say "I have 5 currently". 
            // Let's implement 'ajuste' as "Add/Subtract" for now to keep it simple with the 'quantity' field.
            // If user wants to set exact stock, they calculate difference. 
            // actually, let's treat 'ajuste' as a signed manual change. 
            // But the dropdown usually says "Add stock" or "Remove stock". 
            // Let's stick to 'entrada' (add) and 'salida' (remove). 'ajuste' might be used for "Lost/Found".
            // Let's handle 'ajuste' as an addition/subtraction based on a sign? 
            // NO, let's just use 'entrada' and 'salida' for simplicity in UI.
            // If type is 'ajuste' and quantity is positive, we add? 
            // Let's assume 'ajuste' is just a label for 'reason' but logic matches entry/exit. 
            // Actually, let's deprecate 'ajuste' logic here and just use entry/exit types for the math.

            // Wait, if I want to correct stock from 10 to 12. I add 2. Type=entrada, Reason=Correction.
            // If I want to correct 10 to 8. I remove 2. Type=salida, Reason=Correction.

            // So logic is simple:
            if ($request->type === 'ajuste') {
                // For now, fail or treat as addition? Let's just default to addition.
                $newStock = $product->stock + $quantity;
            } else {
                $newStock = $product->stock; // Should not happen
            }
        }

        $product->update(['stock' => $newStock]);

        InventoryMovement::create([
            'product_id' => $product->id,
            'user_id' => Auth::id(),
            'type' => $request->type,
            'quantity' => $quantity, // We store absolute quantity
            'current_stock_snapshot' => $newStock,
            'reason' => $request->reason,
        ]);

        return redirect()->route('inventory.index')->with('success', 'Movimiento de inventario registrado.');
    }
}
