<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use App\Models\ActivityLog;
use App\Models\User;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        $users = User::where('is_active', true)
            ->with('role')
            ->orderBy('name')
            ->get()
            ->map(fn($u) => [
                'id' => $u->id,
                'name' => $u->name,
                'has_pin' => $u->hasPin(),
            ]);

        return view('auth.login', compact('users'));
    }

    public function login(Request $request)
    {
        $request->validate([
            'user_id' => ['required', 'exists:users,id'],
        ]);

        $user = User::findOrFail($request->user_id);

        if (!$user->is_active) {
            throw ValidationException::withMessages([
                'user_id' => ['Esta cuenta está desactivada.'],
            ]);
        }

        $authenticated = false;

        // Try PIN first if provided and user has a PIN
        if ($request->filled('pin') && $user->hasPin()) {
            if (Hash::check($request->pin, $user->pin)) {
                $authenticated = true;
            } else {
                throw ValidationException::withMessages([
                    'pin' => ['PIN incorrecto.'],
                ]);
            }
        } elseif ($request->filled('password')) {
            // Fall back to password
            if (Hash::check($request->password, $user->password)) {
                $authenticated = true;
            } else {
                throw ValidationException::withMessages([
                    'password' => [__('auth.failed')],
                ]);
            }
        } else {
            throw ValidationException::withMessages([
                'pin' => ['Debe ingresar su PIN o contraseña.'],
            ]);
        }

        if ($authenticated) {
            Auth::login($user, $request->boolean('remember'));
            $request->session()->regenerate();

            // Log Activity
            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'login',
                'description' => 'User logged in',
                'ip_address' => $request->ip(),
            ]);

            // Redirect based on role
            if (Auth::user()->isAdmin()) {
                return redirect()->intended('admin/dashboard');
            }

            return redirect()->intended('caja/operations');
        }

        throw ValidationException::withMessages([
            'user_id' => [__('auth.failed')],
        ]);
    }

    public function logout(Request $request)
    {
        if (Auth::check()) {
            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'logout',
                'description' => 'User logged out',
                'ip_address' => $request->ip(),
            ]);
        }

        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
}
